<?php
require_once _PS_MODULE_DIR_ . 'mtborica/mtborica.php';

/**
 * Class MtboricaValidateAddToCartModuleFrontController
 *
 * Handles AJAX validation before adding a product to cart.
 * Validates that recurring and standard products are not mixed,
 * and that all recurring products use the same plan.
 */
class MtboricaValidateAddToCartModuleFrontController extends ModuleFrontController
{
    /**
     * Aggregated JSON payload to be returned by displayAjax().
     *
     * Structure on success: {
     *   success: true,
     *   allowed: bool,        // true if product can be added
     *   message?: string     // optional message
     * }
     *
     * Structure on failure: {
     *   success: true,
     *   allowed: false,
     *   message: string      // error message explaining why
     * }
     *
     * @var array|null
     */
    public $result = null;

    /**
     * Initializes controller content for AJAX flow.
     *
     * Responsibilities:
     * - Accept POST requests with product_id and recurring_plan_id
     * - Validate cart contents against the product being added
     * - Return JSON result indicating if product can be added
     *
     * @return void
     */
    public function initContent()
    {
        parent::initContent();
        $this->ajax = true;

        // Check if recurring feature is enabled
        if (!Mtborica::isRecurringEnabled()) {
            $this->result = [
                'success' => true,
                'allowed' => true, // If recurring is disabled, allow all additions
            ];
            return;
        }

        $this->result = [];

        if (isset($_SERVER['REQUEST_METHOD']) && 'POST' === $_SERVER['REQUEST_METHOD']) {
            // Get product ID and selected recurring plan ID
            $product_id = (int) Tools::getValue('product_id', 0);
            $recurring_plan_id = (int) Tools::getValue('recurring_plan_id', 0);
            $id_product_attribute = (int) Tools::getValue('id_product_attribute', 0);

            if ($product_id <= 0) {
                $this->result = [
                    'success' => true,
                    'allowed' => false,
                    'message' => $this->module->l('Invalid product.', 'validateAddToCart'),
                ];
                return;
            }

            // Check if product has recurring plans
            $product_has_recurring = Mtborica::productHasRecurringPlan($product_id);
            $is_adding_recurring = $product_has_recurring && $recurring_plan_id > 0;

            // Get current cart
            $cart = $this->context->cart;
            if (!$cart || !$cart->id) {
                // Empty cart - allow addition
                $this->result = [
                    'success' => true,
                    'allowed' => true,
                ];
                return;
            }

            // Get cart products
            $cart_products = $cart->getProducts();
            if (empty($cart_products)) {
                // Empty cart - allow addition
                $this->result = [
                    'success' => true,
                    'allowed' => true,
                ];
                return;
            }

            // Check if cart has recurring products (products actually added WITH a plan)
            // Use getCartRecurringPlanIds() which directly queries the database
            $cart_recurring_plan_ids = Mtborica::getCartRecurringPlanIds($cart->id);
            $cart_has_recurring = !empty($cart_recurring_plan_ids);

            // Check if cart has standard products (products added WITHOUT a plan)
            // A product that CAN have a plan but was added WITHOUT a plan is also considered standard
            $cart_has_standard = false;
            foreach ($cart_products as $cart_product) {
                $cart_product_id = (int) $cart_product['id_product'];
                $cart_product_attribute_id = (int) ($cart_product['id_product_attribute'] ?? 0);
                
                // Check if product was added WITHOUT a plan
                $plan_id = Mtborica::getCartProductRecurringPlan($cart->id, $cart_product_id, $cart_product_attribute_id);
                if ($plan_id <= 0) {
                    // Product was added without a plan - it's a standard product
                    $cart_has_standard = true;
                    break;
                }
            }

            // Validation rules:
            // 1. Cannot mix recurring and standard products
            if ($cart_has_standard && $is_adding_recurring) {
                $this->result = [
                    'success' => true,
                    'allowed' => false,
                    'message' => $this->module->l('You cannot mix recurring payment products with standard products. Please remove standard products from your cart first.', 'validateAddToCart'),
                ];
                return;
            }

            if ($cart_has_recurring && !$is_adding_recurring) {
                $this->result = [
                    'success' => true,
                    'allowed' => false,
                    'message' => $this->module->l('You cannot add standard products to a cart with recurring payment products. Please remove recurring products from your cart first.', 'validateAddToCart'),
                ];
                return;
            }

            // 2. If adding recurring product, check if it matches existing recurring plan
            if ($is_adding_recurring && $cart_has_recurring) {
                // Check if the product being added has the selected plan in its available plans
                $product_plan_ids = Mtborica::getProductRecurringPlanIds($product_id);
                if (!in_array($recurring_plan_id, $product_plan_ids)) {
                    $this->result = [
                        'success' => true,
                        'allowed' => false,
                        'message' => $this->module->l('The selected recurring plan is not available for this product.', 'validateAddToCart'),
                    ];
                    return;
                }

                // Get all recurring plan IDs from existing cart products by reading from database
                // Use the already fetched plan IDs
                $existing_plan_ids = $cart_recurring_plan_ids;

                // If we have existing plan IDs, check if they all match the new plan
                if (!empty($existing_plan_ids)) {
                    // Check if all existing plans are the same
                    $unique_existing_plans = array_unique($existing_plan_ids);
                    if (count($unique_existing_plans) > 1) {
                        // Cart already has products with different plans - deny
                        $this->result = [
                            'success' => true,
                            'allowed' => false,
                            'message' => $this->module->l('Your cart contains products with different recurring payment plans. Please remove products with different plans first.', 'validateAddToCart'),
                        ];
                        return;
                    }

                    // Get the single existing plan ID
                    $existing_plan_id = reset($unique_existing_plans);

                    // If cart has recurring products with a different plan, deny
                    if ($recurring_plan_id !== $existing_plan_id) {
                        $this->result = [
                            'success' => true,
                            'allowed' => false,
                            'message' => $this->module->l('All products in your cart must use the same recurring payment plan. Please remove products with different plans or select the same plan for this product.', 'validateAddToCart'),
                        ];
                        return;
                    }
                }
            }

            // All validations passed - allow addition
            $this->result = [
                'success' => true,
                'allowed' => true,
            ];
        } else {
            // Not a POST request
            $this->result = [
                'success' => false,
                'allowed' => false,
                'message' => $this->module->l('Invalid request method.', 'validateAddToCart'),
            ];
        }
    }

    /**
     * Outputs the aggregated JSON response and terminates execution.
     *
     * @return void
     */
    public function displayAjax()
    {
        header('Content-Type: application/json');
        die(Tools::jsonEncode($this->result));
    }
}
