<?php
/**
 * Class MtboricaUpdateRecurringModuleFrontController
 *
 * Updates an existing recurring plan. Validates and sanitizes input
 * values according to `mtborica_recurring` table schema and persists
 * via `MtboricaRecurring` ObjectModel. Responds with JSON suitable
 * for the admin UI.
 */
class MtboricaUpdateRecurringModuleFrontController extends ModuleFrontController
{
    public $result = array();

    /**
     * Initializes controller and enforces admin authentication.
     *
     * @return void
     */
    public function init()
    {
        parent::init();

        $cookie = new Cookie('psAdmin');
        if (!$cookie->id_employee || !$cookie->isLoggedBack()) {
            die(json_encode(array(
                'success' => false,
                'message' => 'You must be logged in as administrator'
            )));
        }
    }

    /**
     * Processes POST update request for a recurring plan.
     *
     * Input: POST `id`, `name`, `duration`, `duration_unit`,
     *        `frequency`, `frequency_unit`, `mday_payment`
     * Output: JSON with `success`, optional `errors` or `message`
     *
     * @return void
     */
    public function initContent()
    {
        parent::initContent();

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->result = array(
                'success' => false,
                'message' => $this->l('Invalid request method.')
            );
            return;
        }

        $id = (int) Tools::getValue('id');
        if ($id <= 0) {
            $this->result = array(
                'success' => false,
                'message' => $this->l('Invalid ID parameter.')
            );
            return;
        }

        require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaRecurring.php';
        $recurring = new MtboricaRecurring($id);
        if (!Validate::isLoadedObject($recurring)) {
            $this->result = array(
                'success' => false,
                'message' => $this->l('Recurring plan not found.')
            );
            return;
        }

        $errors = array();

        $name = (string) Tools::getValue('name', '');
        $duration = (string) Tools::getValue('duration', '');
        $duration_unit = (string) Tools::getValue('duration_unit', '');
        $frequency = (string) Tools::getValue('frequency', '');
        $frequency_unit = (string) Tools::getValue('frequency_unit', '');
        $mday_payment = (string) Tools::getValue('mday_payment', '');

        // name
        $name = trim($name);
        if ($name === '') {
            $errors['name'] = $this->l('Name is required.');
        } elseif (Tools::strlen($name) > 128) {
            $name = Tools::substr($name, 0, 128);
        } elseif (!Validate::isCleanHtml($name)) {
            $errors['name'] = $this->l('Invalid name.');
        }

        // allowed units
        $allowedUnits = array('D', 'M', 'Q', 'H', 'Y');

        // duration
        $duration = trim($duration);
        if ($duration === '' || !ctype_digit($duration)) {
            $errors['duration'] = $this->l('Duration must be a positive integer.');
        } else {
            $durationInt = (int) $duration;
            if ($durationInt < 1) {
                $errors['duration'] = $this->l('Duration must be at least 1.');
            }
            if (Tools::strlen($duration) > 4) {
                $errors['duration'] = $this->l('Duration must be 4 digits or fewer.');
            }
        }

        // duration_unit
        $duration_unit = Tools::strtoupper(trim($duration_unit));
        if (!in_array($duration_unit, $allowedUnits)) {
            $duration_unit = 'D';
        }

        // frequency
        $frequency = trim($frequency);
        if ($frequency === '' || !ctype_digit($frequency)) {
            $errors['frequency'] = $this->l('Frequency must be a positive integer.');
        } else {
            $frequencyInt = (int) $frequency;
            if ($frequencyInt < 1) {
                $errors['frequency'] = $this->l('Frequency must be at least 1.');
            }
            if (Tools::strlen($frequency) > 4) {
                $errors['frequency'] = $this->l('Frequency must be 4 digits or fewer.');
            }
        }

        // frequency_unit
        $frequency_unit = Tools::strtoupper(trim($frequency_unit));
        if (!in_array($frequency_unit, $allowedUnits)) {
            $frequency_unit = 'D';
        }

        // mday_payment 1..31
        $mday_payment = trim($mday_payment);
        $mday_payment = trim($mday_payment);
        if ($frequency_unit === 'M' && $mday_payment !== '') {
            if (!preg_match('/^\d{1,2}$/', $mday_payment) || (int) $mday_payment <= 0 || (int) $mday_payment > 31) {
                $errors['mday_payment'] = $this->l('Please enter a valid value for the Mday payment field. Please enter a valid value for the MDay parameter field(from 1 to 31).');
            }
        } else if ($frequency_unit !== 'M' && $mday_payment !== '') {
            $errors['mday_payment'] = $this->l('If the Frequency Unit field is not month, you cannot enter a value in the Mday Payment field.');
        }

        if (!empty($errors)) {
            $this->result = array(
                'success' => false,
                'errors' => $errors,
            );
            return;
        }

        $recurring->name = $name;
        $recurring->recur_duration = $duration;
        $recurring->recur_duration_unit = $duration_unit;
        $recurring->recur_freq = $frequency;
        $recurring->recur_freq_unit = $frequency_unit;
        $recurring->recur_mday_payment = $mday_payment;

        if (!$recurring->save()) {
            $this->result = array(
                'success' => false,
                'message' => $this->l('Failed to update recurring plan.'),
            );
            return;
        }

        $this->result = array(
            'success' => true,
            'message' => $this->l('Recurring plan updated successfully.'),
            'id' => (int) $recurring->id,
        );
    }

    /**
     * Outputs the JSON response.
     *
     * @return void
     */
    public function displayAjax()
    {
        header('Content-Type: application/json');
        die(Tools::jsonEncode($this->result));
    }
}


