<?php
require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaRecurring.php';

/**
 * Class MtboricaCreateReccuringModuleFrontController
 *
 * Handles AJAX creation of recurring payment plans from the module's admin UI.
 * Performs admin authentication using the back-office cookie, validates and
 * sanitizes inputs according to the schema of table `mtborica_recurring`,
 * persists data via the `MtboricaRecurring` ObjectModel and returns a JSON
 * response suitable for consumption by JavaScript.
 */
class MtboricaCreateReccuringModuleFrontController extends ModuleFrontController
{
    /**
     * Aggregated JSON payload to be returned by displayAjax().
     *
     * Structure on success: {
     *   success: true,
     *   id: int,             // newly created recurring id
     *   message: string      // human-readable success message
     * }
     *
     * Structure on failure: {
     *   success: false,
     *   message?: string,    // error summary (optional)
     *   errors?: array       // associative validation errors
     * }
     *
     * @var array|null
     */
    public $result = null;

    /**
     * Initializes controller content for AJAX flow.
     *
     * Responsibilities:
     * - Enforce back-office admin authentication via `psAdmin` cookie
     * - Accept only POST requests for creation
     * - Validate and sanitize: name, duration, duration_unit, frequency,
     *   frequency_unit, mday_payment
     * - Create new `MtboricaRecurring` record and return JSON result
     *
     * @return void
     */
    public function initContent()
    {
        parent::initContent();
        $this->ajax = true;

        // Check if user is logged in admin - use cookie-based check
        $cookie = new Cookie('psAdmin');
        if (!$cookie->id_employee || !$cookie->isLoggedBack()) {
            die(json_encode(array(
                'error' => true,
            )));
        }

        $this->result = array();

        if (isset($_SERVER['REQUEST_METHOD']) && 'POST' === $_SERVER['REQUEST_METHOD']) {
            $errors = array();

            // Raw inputs
            $name = (string) Tools::getValue('name', '');
            $duration = (string) Tools::getValue('duration', '');
            $duration_unit = (string) Tools::getValue('duration_unit', '');
            $frequency = (string) Tools::getValue('frequency', '');
            $frequency_unit = (string) Tools::getValue('frequency_unit', '');
            $mday_payment = (string) Tools::getValue('mday_payment', '');

            // Sanitization and validation according to DB schema
            // name: VARCHAR(128) NOT NULL
            $name = trim($name);
            if (Tools::strlen($name) === 0) {
                $errors['name'] = $this->l('Name is required.');
            } else {
                if (Tools::strlen($name) > 128) {
                    $name = Tools::substr($name, 0, 128);
                }
                if (!Validate::isCleanHtml($name)) {
                    $errors['name'] = $this->l('Invalid name.');
                }
            }

            // duration: varchar(4) NOT NULL, numeric positive (0-9999)
            $duration = trim($duration);
            if ($duration === '' || !ctype_digit($duration)) {
                $errors['duration'] = $this->l('Duration must be a positive integer.');
            } else {
                $durationInt = (int) $duration;
                if ($durationInt < 1) {
                    $errors['duration'] = $this->l('Duration must be at least 1.');
                }
                if (Tools::strlen($duration) > 4) {
                    $errors['duration'] = $this->l('Duration must be 4 digits or fewer.');
                }
            }

            // duration_unit: ENUM('D','M','Q','H','Y') default 'D'
            $allowedUnits = array('D', 'M', 'Q', 'H', 'Y');
            $duration_unit = Tools::strtoupper(trim($duration_unit));
            if (!in_array($duration_unit, $allowedUnits)) {
                $duration_unit = 'D';
            }

            // frequency: varchar(4) NOT NULL, numeric positive (0-9999)
            $frequency = trim($frequency);
            if ($frequency === '' || !ctype_digit($frequency)) {
                $errors['frequency'] = $this->l('Frequency must be a positive integer.');
            } else {
                $frequencyInt = (int) $frequency;
                if ($frequencyInt < 1) {
                    $errors['frequency'] = $this->l('Frequency must be at least 1.');
                }
                if (Tools::strlen($frequency) > 4) {
                    $errors['frequency'] = $this->l('Frequency must be 4 digits or fewer.');
                }
            }

            // frequency_unit: ENUM('D','M','Q','H','Y') default 'D'
            $frequency_unit = Tools::strtoupper(trim($frequency_unit));
            if (!in_array($frequency_unit, $allowedUnits)) {
                $frequency_unit = 'D';
            }

            // mday_payment: varchar(2) NOT NULL, [1..31]
            $mday_payment = trim($mday_payment);
            if ($frequency_unit === 'M' && $mday_payment !== '') {
                if (!preg_match('/^\d{1,2}$/', $mday_payment) || (int) $mday_payment <= 0 || (int) $mday_payment > 31) {
                    $errors['mday_payment'] = $this->l('Please enter a valid value for the Mday payment field. Please enter a valid value for the MDay parameter field(from 1 to 31).');
                }
            } else if ($frequency_unit !== 'M' && $mday_payment !== '') {
                $errors['mday_payment'] = $this->l('If the Frequency Unit field is not month, you cannot enter a value in the Mday Payment field.');
            }

            if (!empty($errors)) {
                $this->result = array(
                    'success' => false,
                    'errors' => $errors,
                );
                return;
            }

            $recurring = new MtboricaRecurring();
            $recurring->created_at = date('Y-m-d H:i:s');
            $recurring->name = $name;
            $recurring->status = 1;
            $recurring->recur_duration = $duration;
            $recurring->recur_duration_unit = $duration_unit;
            $recurring->recur_freq = $frequency;
            $recurring->recur_freq_unit = $frequency_unit;
            $recurring->recur_mday_payment = $mday_payment;

            $ok = $recurring->save();

            if (!$ok) {
                $this->result = array(
                    'success' => false,
                    // Don't send message - template will use fallback translation
                );
                return;
            }

            $newId = (int) $recurring->id;

            $this->result = array(
                'success' => true,
                'id' => $newId,
                // Don't send message - template will use fallback translation
            );
        }
    }

    /**
     * Outputs the aggregated JSON response and terminates execution.
     *
     * @return void
     */
    public function displayAjax()
    {
        header('Content-Type: application/json');
        die(Tools::jsonEncode($this->result));
    }
}
