<?php

/**
 * Admin Controller for BORICA Order Management
 * Handles payment status checks and reversals
 */

require_once _PS_MODULE_DIR_ . 'mtborica/mtborica.php';
require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaBoricaHelper.php';
require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaOrder.php';
require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaLogger.php';

class AdminMtboricaOrderController extends ModuleAdminController
{

    public function __construct()
    {
        $this->bootstrap = true;
        $this->module = new Mtborica();

        parent::__construct();

        $this->controller_type = 'moduleadmin';
    }

    /**
     * Process AJAX requests for payment actions
     */
    public function init()
    {
        parent::init();

        if (Tools::isSubmit('ajax') && Tools::isSubmit('action_type')) {
            $this->processAjax();
        }
    }

    /**
     * Process AJAX action
     */
    private function processAjax()
    {
        $action_type = Tools::getValue('action_type');
        $token = Tools::getValue('token');

        header('Content-Type: application/json');

        // Add security headers
        // HSTS (HTTP Strict Transport Security) - only if HTTPS is used
        // Check if request is over HTTPS
        $isHttps = (isset($_SERVER['HTTPS']) && ($_SERVER['HTTPS'] === 'on' || $_SERVER['HTTPS'] === '1')) ||
            (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443) ||
            (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https');

        if ($isHttps) {
            header('Strict-Transport-Security: max-age=31536000; includeSubDomains');
        }
        // Additional security headers
        header('X-Content-Type-Options: nosniff');
        header('X-Frame-Options: SAMEORIGIN');

        // Validate CSRF token
        $expectedToken = Tools::getAdminTokenLite('AdminMtboricaOrder');
        if (!$token || $token !== $expectedToken) {
            $this->ajaxDie(json_encode([
                'resultChange' => false,
                'resultChangeTitle' => $this->l('Error'),
                'resultChangeText' => $this->l('Invalid security token')
            ]));
        }

        switch ($action_type) {
            case 'checkPayment':
                $this->checkPaymentStatus();
                break;
            case 'cancelPayment':
                $this->cancelPayment();
                break;
            default:
                $this->ajaxDie(json_encode([
                    'resultChange' => false,
                    'resultChangeTitle' => $this->l('Error'),
                    'resultChangeText' => $this->l('Unknown action')
                ]));
        }
    }

    /**
     * Check payment status with BORICA
     *
     */
    private function checkPaymentStatus()
    {
        // Get and sanitize BORICA parameters from POST request
        $borica_url = filter_var(Tools::getValue('BORICA_URL'), FILTER_SANITIZE_URL);
        $borica_terminal = trim((string) Tools::getValue('TERMINAL'));
        $borica_trtype = (int) Tools::getValue('TRTYPE');
        $borica_order = trim((string) Tools::getValue('ORDER'));
        $borica_tran_trtype = (int) Tools::getValue('TRAN_TRTYPE');
        $borica_nonce = trim((string) Tools::getValue('NONCE'));
        $borica_currency_code = strtoupper(trim((string) Tools::getValue('BORICA_CURRENCY_CODE')));
        $borica_action = trim((string) Tools::getValue('ACTION'));
        $borica_rc = trim((string) Tools::getValue('RC'));
        $borica_status = trim((string) Tools::getValue('STATUSMSG'));

        // Additional validation for URL
        if ($borica_url && !filter_var($borica_url, FILTER_VALIDATE_URL)) {
            $borica_url = null;
        }

        // Validate currency code format (ISO 4217)
        if ($borica_currency_code && !preg_match('/^[A-Z]{3}$/', $borica_currency_code)) {
            $borica_currency_code = null;
        }

        $borica_p_sign_check_payment = MtboricaBoricaHelper::signCheckPayment(
            $borica_terminal,
            $borica_trtype,
            $borica_order,
            $borica_nonce,
            $borica_currency_code
        );
        $borica_p_sign = $borica_p_sign_check_payment['pSign'];

        // Prepare POST data for BORICA API
        $postData = [
            'TERMINAL' => $borica_terminal,
            'TRTYPE' => $borica_trtype,
            'ORDER' => $borica_order,
            'TRAN_TRTYPE' => $borica_tran_trtype,
            'NONCE' => $borica_nonce,
            'P_SIGN' => $borica_p_sign,
        ];

        if (1 === (int) Configuration::get('mtborica_debug')) {
            MtboricaLogger::logCheckPaymentSent($postData);
        }

        // Call BORICA API
        $response = $this->callBoricaApi($borica_url, $postData);

        // Check for cURL errors
        if ($response['error']) {
            $this->ajaxDie(json_encode([
                'resultChange' => false,
                'resultChangeTitle' => $this->l('Error'),
                'resultChangeText' => $this->l('There was an error communicating with the BORICA server.')
            ]));
        }

        // Check HTTP response code
        if ($response['response']['code'] !== 200) {
            $this->ajaxDie(json_encode([
                'resultChange' => false,
                'resultChangeTitle' => $this->l('Error'),
                'resultChangeText' => $this->l('The BORICA server returned an error.')
            ]));
        }

        // Parse response body (BORICA returns JSON data)
        $responseData = json_decode($response['body'], true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->ajaxDie(json_encode([
                'resultChange' => false,
                'resultChangeTitle' => $this->l('Error'),
                'resultChangeText' => $this->l('Invalid response format from BORICA server')
            ]));
        }

        $response_p_sign = isset($responseData['P_SIGN']) ? $responseData['P_SIGN'] : '';
        $response_action = isset($responseData['ACTION']) ? $responseData['ACTION'] : '';
        $response_rc = isset($responseData['RC']) ? $responseData['RC'] : '';
        $response_approval = isset($responseData['APPROVAL']) ? $responseData['APPROVAL'] : '';
        $response_terminal = isset($responseData['TERMINAL']) ? $responseData['TERMINAL'] : '';
        $response_trtype = isset($responseData['TRTYPE']) ? $responseData['TRTYPE'] : '';
        $response_amount = isset($responseData['AMOUNT']) ? $responseData['AMOUNT'] : '';
        $response_currency = isset($responseData['CURRENCY']) ? $responseData['CURRENCY'] : '';
        $response_order = isset($responseData['ORDER']) ? $responseData['ORDER'] : '';
        $response_rrn = isset($responseData['RRN']) ? $responseData['RRN'] : '';
        $response_int_ref = isset($responseData['INT_REF']) ? $responseData['INT_REF'] : '';
        $response_pares_status = isset($responseData['PARES_STATUS']) ? $responseData['PARES_STATUS'] : '';
        $response_eci = isset($responseData['ECI']) ? $responseData['ECI'] : '';
        $response_timestamp = isset($responseData['TIMESTAMP']) ? $responseData['TIMESTAMP'] : '';
        $response_nonce = isset($responseData['NONCE']) ? $responseData['NONCE'] : '';
        $response_status = isset($responseData['STATUSMSG']) ? $responseData['STATUSMSG'] : '';

        if (1 === (int) Configuration::get('mtborica_debug')) {
            MtboricaLogger::logCheckPaymentReceived($responseData);
        }

        $borica_check_authorization = MtboricaBoricaHelper::checkAuthorization(
            $response_p_sign,
            $response_action,
            $response_rc,
            $response_approval,
            $response_terminal,
            $response_trtype,
            $response_amount,
            $response_currency,
            $response_order,
            $response_rrn,
            $response_int_ref,
            $response_pares_status,
            $response_eci,
            $response_timestamp,
            $response_nonce
        );

        $is_change =
            $borica_action !== $response_action ||
            $borica_rc !== $response_rc ||
            $borica_status !== trim($response_status);

        $response_action_txt = '';
        switch ($response_action) {
            case '0':
                $response_action_txt = '<span style="color:green;">' . $this->l('Successfully completed transaction') . '</span>';
                break;
            case '1':
                $response_action_txt = '<span style="color:red;">' . $this->l('Duplicate transaction') . '</span>';
                break;
            case '2':
                $response_action_txt = '<span style="color:red;">' . $this->l('Transaction declined') . '</span>';
                break;
            case '3':
                $response_action_txt = '<span style="color:red;">' .
                    $this->l('Error processing transaction') .
                    '</span>';
                break;
            case '7':
                $response_action_txt = '<span style="color:red;">' .
                    $this->l('Duplicate transaction on failed authentication') .
                    '</span>';
                break;
            case '21':
                $response_action_txt = '<span style="color:red;">' . $this->l('Soft Decline') . '</span>';
                break;
            case '999':
                $response_action_txt = '<span style="color:red;">' . $this->l('Transaction not completed') . '</span>';
                break;
        }

        if ($borica_check_authorization) {
            if ($is_change) {
                $data_check = [
                    'action' => '' !== $response_action ? $response_action : '999',
                    'rc' => '' !== $response_rc ? $response_rc : '999',
                    'status' => '' !== $response_status ? $response_status : '999',
                    'approval' => '' !== $response_approval ? $response_approval : '999',
                    'nonce' => $borica_nonce,
                ];
                if (MtboricaOrder::updateBoricaOrder($data_check)) {
                    $this->ajaxDie(json_encode([
                        'resultChange' => true,
                        'resultChangeTitle' => $this->l('Success'),
                        'resultChangeText' => $this->l('The transaction status has changed'),
                        'responseRc' => $response_rc,
                        'responseStatus' => $response_status,
                        'responseActionTxt' => $response_action_txt,
                        'test' => [
                            $borica_action,
                            $response_action,
                            $borica_rc,
                            $response_rc,
                            $borica_status,
                            $response_status,
                            $is_change,
                        ]
                    ]));
                } else {
                    $this->ajaxDie(json_encode([
                        'resultChange' => false,
                        'resultChangeTitle' => $this->l('Error'),
                        'resultChangeText' => $this->l('Failed to update the transaction status')
                    ]));
                }
            } else {
                $this->ajaxDie(json_encode([
                    'resultChange' => true,
                    'resultChangeTitle' => $this->l('Message'),
                    'resultChangeText' => $this->l('There are no changes to the transaction status')
                ]));
            }
        } else {
            $this->ajaxDie(json_encode([
                'resultChange' => false,
                'resultChangeTitle' => $this->l('Error'),
                'resultChangeText' => $this->l('We cannot determine the status of the selected transaction')
            ]));
        }
    }

    /**
     * Reverse payment with BORICA
     *
     * @param int $id_order Order ID
     */
    private function cancelPayment()
    {
        /// Get and sanitize BORICA parameters from POST request
        $borica_url = filter_var(Tools::getValue('BORICA_URL'), FILTER_SANITIZE_URL);
        $borica_terminal = trim((string) Tools::getValue('TERMINAL'));
        $borica_trtype = (int) Tools::getValue('TRTYPE');
        $borica_amount = filter_var(Tools::getValue('AMOUNT'), FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
        $borica_current_amount = filter_var(Tools::getValue('CURRENT_AMOUNT'), FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
        $borica_currency = strtoupper(trim((string) Tools::getValue('CURRENCY')));
        $borica_order = trim((string) Tools::getValue('ORDER'));
        $borica_desc = trim((string) Tools::getValue('DESC'));
        $borica_merchant = trim((string) Tools::getValue('MERCHANT'));
        $borica_merchant_name = trim((string) Tools::getValue('MERCH_NAME'));
        $borica_merchant_url = trim((string) Tools::getValue('MERCH_URL'));
        $borica_email = trim((string) Tools::getValue('EMAIL'));
        $borica_country = trim((string) Tools::getValue('COUNTRY'));
        $borica_merchant_gmt = trim((string) Tools::getValue('MERCH_GMT'));
        $borica_lang = trim((string) Tools::getValue('LANG'));
        $borica_addendum = trim((string) Tools::getValue('ADDENDUM'));
        $borica_custom_order = trim((string) Tools::getValue('AD_CUST_BOR_ORDER_ID'));
        $borica_rrn = trim((string) Tools::getValue('RRN'));
        $borica_int_ref = trim((string) Tools::getValue('INT_REF'));
        $borica_timestamp = trim((string) Tools::getValue('TIMESTAMP'));
        $borica_nonce = trim((string) Tools::getValue('NONCE'));

        // Additional validation for URL
        if ($borica_url && !filter_var($borica_url, FILTER_VALIDATE_URL)) {
            $borica_url = null;
        }

        // Validate currency code format (ISO 4217)
        if ($borica_currency && !preg_match('/^[A-Z]{3}$/', $borica_currency)) {
            $borica_currency = null;
        }

        $borica_p_sign_drop_payment = MtboricaBoricaHelper::signDropPayment(
            $borica_terminal,
            $borica_trtype,
            $borica_current_amount,
            $borica_currency,
            $borica_order,
            $borica_timestamp,
            $borica_nonce
        );
        $borica_p_sign = $borica_p_sign_drop_payment['pSign'];

        // Prepare POST data for BORICA API
        $postData = [
            'TERMINAL' => $borica_terminal,
            'TRTYPE' => $borica_trtype,
            'AMOUNT' => $borica_current_amount,
            'CURRENCY' => $borica_currency,
            'ORDER' => $borica_order,
            'DESC' => $borica_desc,
            'MERCHANT' => $borica_merchant,
            'MERCH_NAME' => $borica_merchant_name,
            'MERCH_URL' => $borica_merchant_url,
            'EMAIL' => $borica_email,
            'COUNTRY' => $borica_country,
            'MERCH_GMT' => $borica_merchant_gmt,
            'LANG' => $borica_lang,
            'ADDENDUM' => $borica_addendum,
            'AD.CUST_BOR_ORDER_ID' => $borica_custom_order,
            'RRN' => $borica_rrn,
            'INT_REF' => $borica_int_ref,
            'TIMESTAMP' => $borica_timestamp,
            'NONCE' => $borica_nonce,
            'P_SIGN' => $borica_p_sign,
        ];

        if (1 === (int) Configuration::get('mtborica_debug')) {
            MtboricaLogger::logDropPaymentSent($postData);
        }

        // Call BORICA API
        $response = $this->callBoricaApi($borica_url, $postData);

        // Check for cURL errors
        if ($response['error']) {
            $this->ajaxDie(json_encode([
                'resultChange' => false,
                'resultChangeTitle' => $this->l('Error'),
                'resultChangeText' => $this->l('There was an error communicating with the BORICA server.')
            ]));
        }

        // Check HTTP response code
        if ($response['response']['code'] !== 200) {
            $this->ajaxDie(json_encode([
                'resultChange' => false,
                'resultChangeTitle' => $this->l('Error'),
                'resultChangeText' => $this->l('The BORICA server returned an error.')
            ]));
        }

        // Parse response body (BORICA returns JSON data)
        $responseData = json_decode($response['body'], true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->ajaxDie(json_encode([
                'resultChange' => false,
                'resultChangeTitle' => $this->l('Error'),
                'resultChangeText' => $this->l('Invalid response format from BORICA server')
            ]));
        }

        $response_p_sign = isset($responseData['P_SIGN']) ? $responseData['P_SIGN'] : '';
        $response_action = isset($responseData['ACTION']) ? $responseData['ACTION'] : '';
        $response_rc = isset($responseData['RC']) ? $responseData['RC'] : '';
        $response_approval = isset($responseData['APPROVAL']) ? $responseData['APPROVAL'] : '';
        $response_terminal = isset($responseData['TERMINAL']) ? $responseData['TERMINAL'] : '';
        $response_trtype = isset($responseData['TRTYPE']) ? $responseData['TRTYPE'] : '';
        $response_amount = isset($responseData['AMOUNT']) ? $responseData['AMOUNT'] : '';
        $response_currency = isset($responseData['CURRENCY']) ? $responseData['CURRENCY'] : '';
        $response_order = isset($responseData['ORDER']) ? $responseData['ORDER'] : '';
        $response_rrn = isset($responseData['RRN']) ? $responseData['RRN'] : '';
        $response_int_ref = isset($responseData['INT_REF']) ? $responseData['INT_REF'] : '';
        $response_pares_status = isset($responseData['PARES_STATUS']) ? $responseData['PARES_STATUS'] : '';
        $response_eci = isset($responseData['ECI']) ? $responseData['ECI'] : '';
        $response_timestamp = isset($responseData['TIMESTAMP']) ? $responseData['TIMESTAMP'] : '';
        $response_nonce = isset($responseData['NONCE']) ? $responseData['NONCE'] : '';

        if (1 === (int) Configuration::get('mtborica_debug')) {
            MtboricaLogger::logDropPaymentReceived($responseData);
        }

        $borica_drop_authorization = MtboricaBoricaHelper::checkAuthorization(
            $response_p_sign,
            $response_action,
            $response_rc,
            $response_approval,
            $response_terminal,
            $response_trtype,
            $response_amount,
            $response_currency,
            $response_order,
            $response_rrn,
            $response_int_ref,
            $response_pares_status,
            $response_eci,
            $response_timestamp,
            $response_nonce
        );

        $request_cancel_txt = '';
        $borica_order_total = (float) $borica_amount;
        $borica_request_cancel = (float) $response_amount;
        $borica_rest = $borica_order_total - $borica_request_cancel;

        $request_cancel = '11';
        if ($borica_drop_authorization) {
            if (0 == (int) $response_action && '00' == $response_rc) {
                $request_cancel = '00';
                // Find the order
                $order = null;
                $order_row = MtboricaOrder::getByNonce($borica_nonce);
                if ($order_row && !empty($order_row->increment_id)) {
                    $order_id = (int) $order_row->increment_id;
                    $order = new Order($order_id);
                    if (!Validate::isLoadedObject($order)) {
                        $order = null;
                    }
                }

                if ($order) {
                    try {
                        $currency = new Currency((int) $order->id_currency);
                        $iso = $currency ? $currency->iso_code : '';
                        $employeeId = (int) $this->context->employee->id;
                        $customer = new Customer((int) $order->id_customer);

                        // Helper for leaving a service message to the order
                        $leaveOrderNote = function (string $text) use ($order) {
                            $m = new Message();
                            $m->id_order = (int) $order->id;
                            $m->message = $text;
                            $m->private = 1; // only visible in admin
                            // in 1.7 it is not mandatory, but it does not hurt:
                            $m->add();
                        };

                        // ---------- PARTIAL REFUND ----------
                        if ($borica_request_cancel > 0 && $borica_request_cancel < $borica_order_total) {
                            // 1) Leave a note in the order
                            $leaveOrderNote(sprintf(
                                'BORICA partial refund: %.2f %s of total %.2f %s. Remaining amount: %.2f %s.',
                                $borica_request_cancel,
                                $iso,
                                $borica_order_total,
                                $iso,
                                $borica_rest,
                                $iso
                            ));

                            // 2) If possible, create a credit slip (OrderSlip) "by amount".
                            // Note: In some minor versions, the method for refund by amount is not publicly available.
                            // Therefore, we check and use the safe path.
                            $creditSlipDone = false;

                            if (class_exists('OrderSlip')) {
                                // In some 1.7.x, it works like this (by amount, without product lines):
                                // OrderSlip::createOrderSlip($order, $product_list, $shipping_cost, $amount, $shipping_amount)
                                // We will try with an empty product_list, only with amount:
                                try {
                                    if (method_exists('OrderSlip', 'createOrderSlip')) {
                                        $product_list = [];               // without product lines
                                        $shipping_cost = false;           // without shipping
                                        $amount = (float) $borica_request_cancel;
                                        $shipping_amount = 0.0;

                                        OrderSlip::createOrderSlip($order, $product_list, $shipping_cost, $amount, $shipping_amount);
                                        $creditSlipDone = true;
                                    }
                                } catch (Exception $e) {
                                    // ignore, we will leave only a service note
                                }
                            }

                            if (!$creditSlipDone) {
                                // Fallback: only a note + (if desired) log
                                // TODO: Implement logging to the database
                            }

                            // 3) If desired, you can add an intermediate status "Partially refunded", if you have one.
                            // If there is a status with code (for example) PS_OS_PARTIAL_REFUND or personal:
                            $partialRefundStateId = (int) Configuration::get('PS_OS_PARTIAL_REFUND'); // ако съществува
                            if ($partialRefundStateId) {
                                $history = new OrderHistory();
                                $history->id_order = (int) $order->id;
                                $history->changeIdOrderState($partialRefundStateId, $order, true);
                                $history->addWithemail(true);
                            }
                            // ---------- FULL REFUND (CANCELLATION) ----------
                        } else {
                            // 1) Leave a note
                            $leaveOrderNote(sprintf(
                                'BORICA full refund: %.2f %s. The order is marked as cancelled/reversed.',
                                $borica_request_cancel,
                                $iso
                            ));

                            // 2) Change status – by request you want Cancel for full refund
                            // In PrestaShop the standard "Refuned" is PS_OS_REFUND, "Canceled" is PS_OS_CANCELED.
                            $canceledStateId = (int) Configuration::get('PS_OS_CANCELED');
                            if ($canceledStateId) {
                                $history = new OrderHistory();
                                $history->id_order = (int) $order->id;
                                $history->changeIdOrderState($canceledStateId, $order, true);
                                $history->addWithemail(true);
                            } else {
                                // fallback: if for some reason there is no configured Canceled – we use Refunded
                                $refundStateId = (int) Configuration::get('PS_OS_REFUND');
                                if ($refundStateId) {
                                    $history = new OrderHistory();
                                    $history->id_order = (int) $order->id;
                                    $history->changeIdOrderState($refundStateId, $order, true);
                                    $history->addWithemail(true);
                                }
                            }
                        }
                    } catch (Exception $e) {
                        // Do nothing
                    }
                }
            }
        }

        // Get currency - check if order exists first
        $currency = null;
        $borica_currency_symbol = '';
        if (isset($order) && $order && Validate::isLoadedObject($order)) {
            try {
                $currency = new Currency((int) $order->id_currency);
                if ($currency && Validate::isLoadedObject($currency)) {
                    $borica_currency_symbol = !empty($currency->sign) ? $currency->sign : (property_exists($currency, 'symbol') && !empty($currency->symbol) ? $currency->symbol : '');
                }
            } catch (Exception $e) {
                // Do nothing
            }
        }
        switch ($request_cancel) {
            case '00':
                if (0 === $borica_rest) {
                    $request_cancel_txt =
                        '<span style="color:green;">' .
                        $this->l('Successful cancellation of payment (or refund).') .
                        ' ' .
                        $this->l('Net amount paid by card') .
                        ' ' .
                        number_format($borica_rest, 2, '.', '') .
                        ' ' .
                        $borica_currency_symbol .
                        '</span>';
                } else {
                    $request_cancel_txt =
                        '<span style="color:green;">' .
                        $this->l('Amount successfully canceled') .
                        ' ' .
                        number_format($borica_rest, 2, '.', '') .
                        ' ' .
                        $borica_currency_symbol .
                        ' ' .
                        $this->l('Net amount paid by card') .
                        ' ' .
                        number_format($borica_request_cancel, 2, '.', '') .
                        ' ' .
                        $borica_currency_symbol .
                        '</span>';
                }
                break;
            case '11':
                $request_cancel_txt =
                    '<span style="color:red;">' .
                    $this->l('Payment cancellation request sent. The request has been rejected.') .
                    ' ' .
                    $this->l('The request was to cancel the amount of:') .
                    ' ' .
                    number_format($borica_rest, 2, '.', '') .
                    '. ' .
                    $this->l('In the event of an unsuccessful cancellation, the merchant may contact the servicing financial institution.') .
                    '</span>';
                break;
        }

        $data_drop = [
            'request_cancel' => $request_cancel,
            'cancel_amount' => $response_amount,
            'nonce' => $borica_nonce,
        ];

        MtboricaOrder::updateBoricaOrder($data_drop);

        $this->ajaxDie(json_encode([
            'resultChange' => true,
            'requestCancelTxt' => $request_cancel_txt,
            'resultChangeTitle' => $this->l('Message'),
            'resultChangeText' => $this->l('The transaction status has changed'),
            'requestCancel' => $request_cancel,
            'cancelAmount' => $response_amount,
            'boricaNonce' => $borica_nonce,
        ]));
    }

    /**
     * Call BORICA API using cURL
     * 
     * @param string $url BORICA API endpoint URL
     * @param array $postData POST parameters to send
     * @return array Response array with structure similar to wp_remote_post:
     *               - 'body' => response body
     *               - 'response' => ['code' => HTTP code, 'message' => status message]
     *               - 'error' => cURL error message or null
     */
    private function callBoricaApi($url, $postData)
    {
        // Encode POST data
        $postDataEncoded = http_build_query($postData);

        // Initialize cURL
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $postDataEncoded,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 20,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/x-www-form-urlencoded',
                'Accept: application/json',
            ],
        ]);

        // Execute request
        $responseBody = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        // Format response similar to wp_remote_post structure
        return [
            'body' => $responseBody,
            'response' => [
                'code' => $httpCode,
                'message' => $httpCode === 200 ? 'OK' : 'Error',
            ],
            'error' => $curlError ? $curlError : null,
        ];
    }
}
