<?php

/**
 * Class MtboricaRecurring
 *
 * ObjectModel representing recurring payment plans stored in
 * database table `mtborica_recurring`.
 *
 * Fields map:
 * - id (int): Primary key
 * - created_at (string|null): Creation timestamp (Y-m-d H:i:s) or null
 * - name (string): Human-readable plan name (max 128)
 * - status (int): 1 = active, 0 = inactive
 * - recur_duration (string): Duration length (numeric string, up to 4)
 * - recur_duration_unit (string): One of D, M, Q, H, Y
 * - recur_freq (string): Frequency length (numeric string, up to 4)
 * - recur_freq_unit (string): One of D, M, Q, H, Y
 * - recur_mday_payment (string): Day of month for payment (1..31 as string)
 *
 * @property int         $id
 * @property string|null $created_at
 * @property string      $name
 * @property int         $status
 * @property string      $recur_duration
 * @property string      $recur_duration_unit
 * @property string      $recur_freq
 * @property string      $recur_freq_unit
 * @property string      $recur_mday_payment
 */
class MtboricaRecurring extends ObjectModel
{
    public $id;
    public $created_at;
    public $name;
    public $status;
    public $recur_duration;
    public $recur_duration_unit;
    public $recur_freq;
    public $recur_freq_unit;
    public $recur_mday_payment;

    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'mtborica_recurring',
        'primary' => 'id',
        'fields' => array(
            'created_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateFormat', 'required' => false, 'allow_null' => true),
            'name' => array('type' => self::TYPE_STRING, 'validate' => 'isCleanHtml', 'required' => true, 'size' => 128),
            'status' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'recur_duration' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 4),
            'recur_duration_unit' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 1),
            'recur_freq' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 4),
            'recur_freq_unit' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 1),
            'recur_mday_payment' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => false, 'size' => 2),
        ),
    );

    /**
     * Get all recurring payments
     *
     * @param bool $active_only Get only active recurring payments
     * @return array
     */
    /**
     * Returns all recurring plans, optionally only active ones.
     *
     * @param bool $active_only When true, returns only records with status = 1
     * @return array Array of associative rows from database
     */
    public static function getAll($active_only = false)
    {
        $sql = 'SELECT * FROM `' . _DB_PREFIX_ . 'mtborica_recurring`';

        if ($active_only) {
            $sql .= ' WHERE `status` = 1';
        }

        $sql .= ' ORDER BY `created_at` DESC';

        $result = Db::getInstance()->executeS($sql);

        if (!$result) {
            return array();
        }

        return $result;
    }

    /**
     * Get recurring payment by ID
     *
     * @param int $id
     * @return MtboricaRecurring|false
     */
    /**
     * Returns a recurring plan hydrated as ObjectModel for given id.
     *
     * @param int $id Recurring plan id
     * @return MtboricaRecurring|false Loaded object or false if not found/invalid id
     */
    public static function getById($id)
    {
        $id = (int) $id;
        if ($id <= 0) {
            return false;
        }

        $sql = 'SELECT * FROM `' . _DB_PREFIX_ . 'mtborica_recurring` WHERE `id` = ' . $id;
        $result = Db::getInstance()->getRow($sql);

        if (!$result) {
            return false;
        }

        $recurring = new self();
        $recurring->hydrate($result);
        return $recurring;
    }

    /**
     * Toggle status of recurring payment
     *
     * @param int $id
     * @return bool
     */
    /**
     * Toggles the active status of a recurring plan.
     *
     * @param int $id Recurring plan id
     * @return bool True on success, false on failure
     */
    public static function toggleRecurringStatus($id)
    {
        $id = (int) $id;
        if ($id <= 0) {
            return false;
        }

        $recurring = new self($id);
        if (!Validate::isLoadedObject($recurring)) {
            return false;
        }

        $recurring->status = 1 - $recurring->status;
        return $recurring->save();
    }

    /**
     * Delete recurring payment
     *
     * @param int $id
     * @return bool
     */
    /**
     * Deletes a recurring plan by id.
     *
     * @param int $id Recurring plan id
     * @return bool True on success, false on failure
     */
    public static function deleteRecurring($id)
    {
        $id = (int) $id;
        if ($id <= 0) {
            return false;
        }

        $recurring = new self($id);
        if (!Validate::isLoadedObject($recurring)) {
            return false;
        }

        return $recurring->delete();
    }

    /**
     * Get active recurring payments count
     *
     * @return int
     */
    /**
     * Returns the count of active recurring plans.
     *
     * @return int Active records count
     */
    public static function getActiveCount()
    {
        $sql = 'SELECT COUNT(*) FROM `' . _DB_PREFIX_ . 'mtborica_recurring` WHERE `status` = 1';
        return (int) Db::getInstance()->getValue($sql);
    }

    /**
     * Get formatted duration string
     *
     * @return string
     */
    /**
     * Returns a human-friendly representation of the plan duration
     * combining `recur_duration` and `recur_duration_unit`.
     *
     * @return string Formatted duration string
     */
    public function getFormattedDuration()
    {
        $units = array(
            'D' => 'day(s)',
            'M' => 'month(s)',
            'Q' => 'quarter(s)',
            'H' => '6-month period(s)',
            'Y' => 'year(s)'
        );

        $unit = isset($units[$this->recur_duration_unit]) ? $units[$this->recur_duration_unit] : 'day(s)';
        return $this->recur_duration . ' ' . $unit;
    }

    /**
     * Get formatted frequency string
     *
     * @return string
     */
    /**
     * Returns a human-friendly representation of the execution frequency
     * combining `recur_freq` and `recur_freq_unit`.
     *
     * @return string Formatted frequency string
     */
    public function getFormattedFrequency()
    {
        $units = array(
            'D' => 'day(s)',
            'M' => 'month(s)',
            'Q' => 'quarter(s)',
            'H' => '6-month period(s)',
            'Y' => 'year(s)'
        );

        $unit = isset($units[$this->recur_freq_unit]) ? $units[$this->recur_freq_unit] : 'day(s)';
        return 'Every ' . $this->recur_freq . ' ' . $unit;
    }
}
